import java.applet.*;
import java.awt.*;
import java.awt.image.*;
import java.net.*;


/**
 * See http://people.netscape.com/fry/java/othello3/ for details.
 * This code isn't commented much, sorry. Please study it well 
 * before sending any questions.
 */
public class Othello extends Applet
{
	static final int NOWHERE = -1;
	
	static final int BLACK = 2;
	static final int WHITE = 1;
	static final int EMPTY = 0;
	static final int OKMOVE = 3;
	
	static final int USER = BLACK;
	static final int COMPUTER = WHITE;
	
	static final int CHANGED = 16;
	
	static final int BOARD_WIDTH = 400;
	static final int BOARD_HEIGHT = 260;
	static final int APPLET_HEIGHT = 345;
	static final int STATUS_BOX_HEIGHT = 50;
	
	static final int ICON_TOP = BOARD_HEIGHT + 30;
	static final int SOUND_LEFT = BOARD_WIDTH - 100;
	static final int RESTART_LEFT = 70;	
	
	static final int A_SPLIT_SECOND = 180;

	int xpos[][], ypos[];

	
	// keep track of where the legal move indicator is
	// also, keep the last one so that it can be redrawn
	int okMoveColumn = NOWHERE;
	int okMoveRow = NOWHERE;
	
	// the zero column is not used, board is 8x8
	int board[][] = new int[9][9];

	Image emptyBoard;
	Image blackBoard;
	Image whiteBoard;
	Image moveBoard;
	
	boolean soundIsTurnedOn = true;
	AudioClip userMoveSound;
	AudioClip computerMoveSound;
	AudioClip pieceFlipSound;
	AudioClip illegalMoveSound;

	Image soundOnIcon;
	Image soundOffIcon;
	Image gameRestartIcon;

	String statusMessage;
	
	// has to be three since USER is 2, COMPUTER is 1
	int score[] = new int[3];

	boolean gameIsOver;			
	

	/**
	 * Set up the board, load everything in so it's speedy
	 */
	public void init()
	{
		// codebase used multiple times, start a new mediatracker
		URL url = getCodeBase();
		MediaTracker tracker = new MediaTracker(this);
			
		// Suck in the images.
		blackBoard = getImage(url, "images/blackboard.gif");
		tracker.addImage(blackBoard, 0);
		whiteBoard = getImage(url, "images/whiteboard.gif");
		tracker.addImage(whiteBoard, 0);
		emptyBoard = getImage(url, "images/emptyboard.gif");
		tracker.addImage(emptyBoard, 0);
		moveBoard = getImage(url, "images/moveboard.gif");
		tracker.addImage(moveBoard, 0);
		soundOnIcon = getImage(url, "images/soundon.gif");
		tracker.addImage(soundOnIcon, 0);
		soundOffIcon = getImage(url, "images/soundoff.gif");
		tracker.addImage(soundOffIcon, 0);
		gameRestartIcon = getImage(url, "images/restart.gif");
		tracker.addImage(gameRestartIcon, 0);
		
		try {
			tracker.waitForAll();
		}
		catch (InterruptedException e) {
		}
		
		// Snatch the audio
		userMoveSound = getAudioClip(url, "audio/pock.au");
		computerMoveSound = getAudioClip(url, "audio/shortlowpock.au");
		pieceFlipSound = getAudioClip(url, "audio/tick.au");
		illegalMoveSound = getAudioClip(url, "audio/bozo.au");
		
		// Setup board regions
		xpos = new int[9][9];
		double left = 50.0;
		double right = 349.0;
		
		for (double j = 0.0; j < 9.0; j++) {
			for (double i = 0.0; i <= 8.0; i++) {
				xpos[(int) i][(int) j] = (int) ((left + (right - left) * (i / 8.0)));
			}
			left -= 5.5;
			right += 5.5;
		}
		
		ypos = new int[9];
		//ypos = { 10, 33, 58, 84, 111, 142, 175, 210, 247 };
		ypos[0] = 10;
		ypos[1] = 33;
		ypos[2] = 58;
		ypos[3] = 84;
		ypos[4] = 111;
		ypos[5] = 142;
		ypos[6] = 175;
		ypos[7] = 210;
		ypos[8] = 247;
		
		beginGame();
	}


	public void beginGame()
	{
		// clear the board
		for (int row = 1; row < 9; row++)
			for (int col = 1; col < 9; col++)
				board[col][row] = EMPTY;
	
		// put the first pieces on the board
		board[4][4] = BLACK; board[4][5] = WHITE;
		board[5][4] = WHITE; board[5][5] = BLACK;

		score[USER] = 2;
		score[COMPUTER] = 2;
		gameIsOver = false;

		statusMessage = "Your turn.";
		paint(this.getGraphics());
	}


	/**
	 * Draw the board, update, whatever
	 */
	public void paint(Graphics g)
	{
		g.clipRect(0, 0, BOARD_WIDTH, APPLET_HEIGHT);
		g.drawImage(emptyBoard, 0, 0, this);
		
		g.setColor(Color.white);
		g.fillRect(0, BOARD_HEIGHT, BOARD_WIDTH, APPLET_HEIGHT - BOARD_HEIGHT);
		
		g.setColor(Color.lightGray);
		g.fillRect(60, BOARD_HEIGHT+20, BOARD_WIDTH-120, STATUS_BOX_HEIGHT);
	
		paintIcons(g);
		paintMessage(g);
		
		// draw everyone
		for (int row = 1; row < 9; row++)
			for (int col = 1; col < 9; col++)
				paintPiece(g, col, row);
	}
	
	
	public void paintIcons(Graphics g)
	{
		// show sound icon
		if (soundIsTurnedOn)
			g.drawImage(soundOnIcon, SOUND_LEFT, ICON_TOP, this );
		else
			g.drawImage(soundOffIcon, SOUND_LEFT, ICON_TOP, this );
	
		// show restart icon
		g.drawImage(gameRestartIcon, RESTART_LEFT, ICON_TOP, this );
	}
	
	
	public void updateIcons()
	{
		Graphics g = this.getGraphics();
		g.clipRect(0, 0, BOARD_WIDTH, APPLET_HEIGHT);
		paintIcons(g);
	}
	
	
	public void paintMessage(Graphics g)
	{
		g.setColor(Color.lightGray);
		g.fillRect(110, BOARD_HEIGHT+20, BOARD_WIDTH-211, STATUS_BOX_HEIGHT);
		
		g.setColor(Color.black);
		g.drawString(statusMessage, 190, BOARD_HEIGHT+50);
		g.drawString("Me: "+score[COMPUTER]+"  You: "+score[USER], 110, BOARD_HEIGHT+50);
	}
	
	
	public void updateMessage(String message)
	{
		Graphics g = this.getGraphics();
		g.clipRect(0, 0, BOARD_WIDTH, APPLET_HEIGHT);
		
		statusMessage = message;
		paintMessage(g);
	}	


	public void waitFor(long duration)
	{
		long stopTime = System.currentTimeMillis() + duration;
		
		do {
		} while (System.currentTimeMillis() < stopTime);
	}


	/**
	 * draw an individual piece
	 */
	public void paintPiece(int col, int row)
	{
		paintPiece(this.getGraphics(), col, row);
	}
	
	
	public void paintPiece(Graphics g, int col, int row)
	{
		boolean pieceFlipped = false;
		
		// changed piece? play flipping sound before redrawing
		if ((board[col][row] & CHANGED) > 0) {
			if (soundIsTurnedOn)
				pieceFlipSound.play();
			board[col][row] -= CHANGED;
			pieceFlipped = true;
		}
 
		// draw graphics for piece based on its status
		g = this.getGraphics();
		g.clipRect(xpos[col-1][row-1], ypos[row-1],
		           xpos[col][row-1] - xpos[col-1][row-1], ypos[row] - ypos[row-1]);
		switch (board[col][row]) {
			case EMPTY:
				g.drawImage(emptyBoard, 0, 0, this );
				break;
			case WHITE:
				g.drawImage(whiteBoard, 0, 0, this );
				break;
			case BLACK:
				g.drawImage(blackBoard, 0, 0, this );
				break;
			case OKMOVE:
				g.drawImage(moveBoard, 0, 0, this );
				break;
		}
		if (pieceFlipped) {
			waitFor(A_SPLIT_SECOND);
		}
	}


	/**
	 * if it's a legitimate move, mark it
	 */
	public boolean mouseMove (Event evt, int x, int y)
	{
		int row = 0, col = 0;
		for (int j = 0; j < 8; j++) {
			if (y > ypos[j] && y < ypos[j+1]) {
				row = j + 1;
				double left = 50 - (row * 5.5);
				double right = 349 + (row * 5.5);
				col = (int) ((x - left) / (right - left) * 8) + 1;
			}
		}

		if (col>8 || row>8 || col<1 || row<1)
			return false;
		
		if (col != okMoveColumn || row != okMoveRow) {
			if (okMoveColumn != NOWHERE) {					// un-mark old piece
				board[okMoveColumn][okMoveRow] = EMPTY;
				paintPiece(okMoveColumn, okMoveRow);
				okMoveColumn = NOWHERE;
				okMoveRow = NOWHERE;
			}
			if (isLegalMove(USER, col, row)) {				// mark the new legal piece
				board[col][row] = OKMOVE;
				paintPiece(col, row);
				okMoveColumn = col;
				okMoveRow = row;
			}
		}
		return true;
	}


	/**
	 * make the user's move, based on the mouse click
	 */
	public boolean mouseDown (Event evt, int x, int y)
	{
		int row = 0, col = 0;
		for (int j = 0; j < 8; j++) {
			if (y > ypos[j] && y < ypos[j+1]) {
				row = j + 1;
				double left = 50 - (row * 5.5);
				double right = 349 + (row * 5.5);
				col = (int) ((x - left) / (right - left) * 8) + 1;
			}
		}

		if (col>8 || row>8 || col<1 || row<1) {
			if (y > ICON_TOP && y < ICON_TOP+32) {
				if (x > SOUND_LEFT && x < SOUND_LEFT+24) {
					soundIsTurnedOn = !soundIsTurnedOn;
					updateIcons();
				} else if (x > RESTART_LEFT && x < RESTART_LEFT+24) {
					beginGame();
				}
			}
			return true;
		}
	
		if (isLegalMove(USER, col, row)) {
 			if (soundIsTurnedOn)
				userMoveSound.play();
		
			// update the old 'ok move' piece
			if (okMoveColumn != NOWHERE)	{
				board[okMoveColumn][okMoveRow] = EMPTY;
				okMoveColumn = okMoveRow = NOWHERE;
			}
			makeMove(USER, col, row);
			doComputerMove();

		} else {
			// not a legal move
			if ( board[col][row] == EMPTY )	{
				if ( soundIsTurnedOn )
					illegalMoveSound.play();
				updateMessage("Illegal move.");
			}
		}
		return true;
	}


	public void doComputerMove()
	{
		boolean canMove = findComputerMove(COMPUTER);
	
		// if the USER has no move, COMPUTER keeps taking moves
		// until the USER has a move, or there is a game over
		
		while (!hasMove(USER) && !gameIsOver) {
			if (canMove) {
				// no USER move, COMPUTER takes another
				canMove = findComputerMove(COMPUTER);
			} else {
				// nothing else to do
				gameIsOver = true;
			}
		}
	
		if (!gameIsOver) {
			updateMessage("Your turn.");
		} else {
			endGame();
		}
	}


	/**
	 * do the best possible move
	 */
	boolean findComputerMove(int color)
	{
		int best_count = 0, this_count,
			best_col = 0, best_row = 0;
			
		for (int col = 1; col < 9; col++) {
			for (int row = 1; row < 9; row++) {
				this_count = rankMove( color, col, row );
				if (best_count < this_count) {
					best_col = col;
					best_row = row;
					best_count = this_count;
				}
			}
		}
	
		if (best_count > 0)	{
			if (soundIsTurnedOn)
				computerMoveSound.play();
			makeMove(color, best_col, best_row);
			return true;
		} else {
			return false;
		}
	}


	/**
	 * score move based on number of pieces flipped
	 * the strategy in here is sad
	 */
	int rankMove (int color, int col, int row)
	{
		int count = 0;
	
		// somebody's piece is already in this location
		if ((board[col][row] != EMPTY) && (board[col][row] != OKMOVE)){
			return 0;
		}

		for (int colinc = -1; colinc < 2; colinc++) {
			for (int rowinc = -1; rowinc < 2; rowinc++) {
				count += flipRow(color, col, row, colinc, rowinc, false);
			}
		}

		if (count > 0) {
			// Make the corners most desirable
			if (((col == 1) || (col == 8)) && ((row == 1) || (row == 8))) {
				count = 64;
			}

			// Make the squares next to the corner least desirable
			if ((((col == 1) || (col == 8)) && ((row == 2) || (row == 7))) ||
				(((row == 1) || (row == 8)) && ((col == 2) || (col == 7))) ||
				(((col == 2) || (col == 7)) && ((row == 2) || (row == 7)))) {
				count = 1;
			}
		}
		return count;
	} 


	/**
	 * make an actual move, put the new piece on the board
	 * flip the rest of the pieces
	 */
	void makeMove (int color, int col, int row)
	{
		score[color] += 1;
		board[col][row] = color;
		for (int colinc = -1; colinc < 2; colinc++) {
			for (int rowinc = -1; rowinc < 2; rowinc++) {
				flipRow(color, col, row, colinc, rowinc, true);
			}
		}
	}


	/**
	 * try and flip a row in a particular direction
	 * if really_flipping, then go ahead and flip 'em
	 * return number of pieces flipped
	 */
	int flipRow (int color, int col, int row, int colinc,
				 int rowinc, boolean really_flipping )
	{
		int newcol = col + colinc;
		int newrow = row + rowinc;
		int opponent = BLACK + WHITE - color;
		int count = 0;
	
		// if not incrementing (moving in a direction), then forget it
		if ((colinc == 0) && (rowinc == 0))
			return 0;
	
		if (newcol == 0 || newcol > 8 || newrow == 0 || newrow > 8)
			return 0;
			
		// if we aren't flipping an opponent, then forget it 
		// have to include the CHANGED in case the screen update
		// has not happened yet (grr.. this is annoying about java..)
		if ((board[newcol][newrow] != opponent) &&
			(board[newcol][newrow] != opponent+CHANGED))
			return 0;
	
		// try to flip as many as possible
		while ((board[newcol][newrow] == opponent) ||
			   (board[newcol][newrow] == opponent+CHANGED)) {
			newcol += colinc;
			newrow += rowinc;
			count++;
			if (newcol == 0 || newcol > 8 || newrow == 0 || newrow > 8)
				return 0;
		}
	
		// is there a matching piece on the other end?
		if (board[newcol][newrow] != color)
			return 0;
	
		// if we're really flipping them now, then do it..
		if (really_flipping) {
			while ((col != newcol) || (row != newrow)) {
				// don't draw if already flipped
				//if (board[col][row] != color) {
				//	board[col][row] = color + CHANGED;
				//	paintPiece(col, row);
				//}
				if (board[col][row] != color)
					board[col][row] = color + CHANGED;
				paintPiece(col, row);
				
				col += colinc;
				row += rowinc;
				score[color] += 1;
				score[opponent] -= 1;
			}
			// ok, so we got a little over-excited
			score[color] -= 1;
			score[opponent] += 1;
		}
		return count;
	}


	// can you even move there?
	boolean isLegalMove (int color, int col, int row)
	{
		if ((board[col][row] != EMPTY) && (board[col][row] != OKMOVE)) {
			return false;
		}
		for (int colinc = -1; colinc < 2; colinc++) {
			for (int rowinc = -1; rowinc < 2; rowinc++) {
				if (flipRow(color, col, row, colinc, rowinc, false) > 0) {
					return true;
				}
			}
		}
		return false;
	}


	/**
	 * does this color have anywhere to go?
	 */
	boolean hasMove (int color)
	{
		for ( int col = 1; col < 9; col++ )	{
			for ( int row = 1; row < 9; row++) {
				if (rankMove(color, col, row) > 0) {
					return true;
				}
			}
		}
		return false;
	}


	/**
	 * end of the match, who wins?
	 */
	void endGame ()
	{
		// USER's a moron, COMPUTER has won
		if ( score[COMPUTER] > score[USER] )	{
			if ( soundIsTurnedOn )
				play( getCodeBase(), "audio/iwin.au");
			updateMessage("I win!");
		}
	
		// USER has > 2 oz. of intelligence
		else if ( score[COMPUTER] < score[USER] )	{
			if ( soundIsTurnedOn )
				play( getCodeBase(), "audio/youwin.au");
			updateMessage("You win!");
		}
		
		// we're all winners. what a happy world.
		else	{
			if ( soundIsTurnedOn )
				play( getCodeBase(), "audio/itsatie.au");
			updateMessage("It's a tie!");
		}
	}
}
